/*
Copyright 2008-2009 Elöd Egyed-Zsigmond, Cyril Laitang
Copyright 2009-2011 Samuel Gesche

This file is part of IPRI News Analyzer.

IPRI News Analyzer is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

IPRI News Analyzer is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with IPRI News Analyzer.  If not, see <http://www.gnu.org/licenses/>.
*/

package proc.rss;

import data.structures.rss.RSSFeed;
import data.structures.rss.RSSFeedInfoExtended;
import data.structures.rss.BaseRSSItem;

import proc.text.Codecs;
import proc.text.XMLCleaner;
import proc.text.Out;

import com.sun.syndication.feed.synd.SyndEntryImpl;
import com.sun.syndication.feed.synd.SyndContentImpl;

import java.io.InputStreamReader;
import java.io.BufferedReader;
import java.io.IOException;

import java.net.URL;
import java.net.MalformedURLException;

import java.util.Arrays;
import java.util.Comparator;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.HashMap;
import java.util.Map;
import java.util.TimeZone;
import java.util.Vector;

import java.util.regex.Pattern;
import java.util.regex.Matcher;

public class RSSParser {

    private static Map<String, Integer> mois;

    static{
        /*String[] id = TimeZone.getAvailableIDs();
        //System.out.println("*****");
        for(int i=0; i<id.length; i++){
            if(id[i].startsWith("p") || id[i].startsWith("P")){
                //System.out.println(id[i]);
            }
        }
        //System.out.println("*****");*/

        mois = new HashMap<String, Integer>();
        mois.put("Jan", new Integer(0));
        mois.put("Janvier", new Integer(0));
        mois.put("January", new Integer(0));
        mois.put("jan", new Integer(0));
        mois.put("janvier", new Integer(0));
        mois.put("january", new Integer(0));
        mois.put("Fev", new Integer(1));
        mois.put("Fév", new Integer(1));
        mois.put("Feb", new Integer(1));
        mois.put("Fevrier", new Integer(1));
        mois.put("Février", new Integer(1));
        mois.put("February", new Integer(1));
        mois.put("fev", new Integer(1));
        mois.put("fév", new Integer(1));
        mois.put("feb", new Integer(1));
        mois.put("fevrier", new Integer(1));
        mois.put("février", new Integer(1));
        mois.put("february", new Integer(1));
        mois.put("Mar", new Integer(2));
        mois.put("Mars", new Integer(2));
        mois.put("March", new Integer(2));
        mois.put("mar", new Integer(2));
        mois.put("Mmars", new Integer(2));
        mois.put("march", new Integer(2));
        mois.put("Avr", new Integer(3));
        mois.put("Apr", new Integer(3));
        mois.put("Avril", new Integer(3));
        mois.put("April", new Integer(3));
        mois.put("avr", new Integer(3));
        mois.put("apr", new Integer(3));
        mois.put("avril", new Integer(3));
        mois.put("april", new Integer(3));
        mois.put("Mai", new Integer(4));
        mois.put("May", new Integer(4));
        mois.put("mai", new Integer(4));
        mois.put("may", new Integer(4));
        mois.put("Jun", new Integer(5));
        mois.put("Juin", new Integer(5));
        mois.put("June", new Integer(5));
        mois.put("jun", new Integer(5));
        mois.put("juin", new Integer(5));
        mois.put("june", new Integer(5));
        mois.put("Jul", new Integer(6));
        mois.put("Juillet", new Integer(6));
        mois.put("July", new Integer(6));
        mois.put("jul", new Integer(6));
        mois.put("juillet", new Integer(6));
        mois.put("july", new Integer(6));
        mois.put("Aou", new Integer(7));
        mois.put("Aoû", new Integer(7));
        mois.put("Aug", new Integer(7));
        mois.put("Aout", new Integer(7));
        mois.put("Août", new Integer(7));
        mois.put("August", new Integer(7));
        mois.put("aou", new Integer(7));
        mois.put("aoû", new Integer(7));
        mois.put("aug", new Integer(7));
        mois.put("aout", new Integer(7));
        mois.put("août", new Integer(7));
        mois.put("august", new Integer(7));
        mois.put("Sep", new Integer(8));
        mois.put("Septembre", new Integer(8));
        mois.put("September", new Integer(8));
        mois.put("sep", new Integer(8));
        mois.put("septembre", new Integer(8));
        mois.put("september", new Integer(8));
        mois.put("Sept", new Integer(8));
        mois.put("sept", new Integer(8));
        mois.put("Oct", new Integer(9));
        mois.put("Octobre", new Integer(9));
        mois.put("October", new Integer(9));
        mois.put("oct", new Integer(9));
        mois.put("octobre", new Integer(9));
        mois.put("october", new Integer(9));
        mois.put("Nov", new Integer(10));
        mois.put("Novembre", new Integer(10));
        mois.put("November", new Integer(10));
        mois.put("nov", new Integer(10));
        mois.put("novembre", new Integer(10));
        mois.put("november", new Integer(10));
        mois.put("Dec", new Integer(11));
        mois.put("Déc", new Integer(11));
        mois.put("Decembre", new Integer(11));
        mois.put("Décembre", new Integer(11));
        mois.put("Décember", new Integer(11));
        mois.put("dec", new Integer(11));
        mois.put("déc", new Integer(11));
        mois.put("decembre", new Integer(11));
        mois.put("décembre", new Integer(11));
        mois.put("décember", new Integer(11));
    }

    public static BaseRSSItem[] lectureFlux(RSSFeedInfoExtended flux) throws RSSParsingException {
        // Vérification de l'URL (certains flux n'en ont pas)

        if(flux.URL.equals("")){
            throw new RSSParsingException("pas de flux");
        }

        // Récupération du XML

        URL url = null;
        try{
            url = new URL(flux.URL);
        } catch(MalformedURLException mue){
            throw new RSSParsingException("erreur dans l'URL "+flux.URL+" : "+mue.getMessage());
        }
        String xml = "";

        BufferedReader data;
        try{
            data = new BufferedReader(new InputStreamReader(url.openStream()));
            String s;
            while((s=data.readLine()) != null){
                xml += s;
            }
            if(xml.trim().length()==0){
                ////System.out.println("rien dans le flux : "+flux.URL);
                throw new RSSParsingException("rien dans le flux "+flux.URL);
            }
            data.close();
        } catch(IOException ioe){
            ////System.out.println("flux illisible : "+flux.URL+" ("+ioe.getMessage()+")");
            return lectureRome(flux);
            //throw new RSSParsingException("flux illisible : "+ioe.getMessage());
        }

        xml = Codecs.desEscapeHTMLText(xml);

        // Récupération des articles

        Pattern item = Pattern.compile("<item[^s].*?/item>");
        Matcher m = item.matcher(xml);
        Vector<String> items = new Vector<String>();
        while(m.find()){
            items.addElement(m.group());
        }
        String[] articles = new String[items.size()];
        items.toArray(articles);

        if(articles.length==0){
            item = Pattern.compile("<entry.*?/entry>");
            m = item.matcher(xml);
            items = new Vector<String>();
            while(m.find()){
                items.addElement(m.group());
            }
            articles = new String[items.size()];
            items.toArray(articles);
        }

        if(articles.length==0){
            ////System.out.println("flux vide : "+flux.URL);
            throw new RSSParsingException("flux vide");
        }

        // Récupération des titres

        String[] titres = new String[articles.length];

        for(int i=0; i<articles.length; i++){
            item = Pattern.compile("<title.*?/title>");
            m = item.matcher(articles[i]);
            String candidat = "";
            if(m.find()){
                candidat = m.group();
            }
            candidat = candidat.replaceAll("<!\\[CDATA\\[", "");
            candidat = candidat.replaceAll("\\]\\]>", "");
            titres[i] = Codecs.deHTMLize(candidat);
        }
        String titre = "";
        for(int i=0; i<titres.length; i++){
            titre += titres[i].trim();
        }
        if(titre.length()==0) {
            ////System.out.println("Titres vide ("+flux.name+" : "+flux.URL+").");
            throw new RSSParsingException("titres vides");
        }

        // Récupération des descriptions

        String[] descriptions = new String[articles.length];

        for(int i=0; i<articles.length; i++){
            item = Pattern.compile("<summary.*?/summary>");
            m = item.matcher(articles[i]);
            String candidat = "";
            if(m.find()){
                candidat = m.group();
            }
            if(candidat.equals("")){
                item = Pattern.compile("<description.*?/description>");
                m = item.matcher(articles[i]);
                if(m.find()){
                    candidat = m.group();
                }
            }
            if(candidat.equals("")){
                item = Pattern.compile("<content.*?/content>");
                m = item.matcher(articles[i]);
                if(m.find()){
                    candidat = m.group();
                }
            }
            if(candidat.equals("")){
                item = Pattern.compile("<atom:summary.*?/atom:summary>");
                m = item.matcher(articles[i]);
                if(m.find()){
                    candidat = m.group();
                }
            }
            candidat = candidat.replaceAll("<!\\[CDATA\\[", "");
            candidat = candidat.replaceAll("\\]\\]>", "");
            descriptions[i] = Codecs.deHTMLize(candidat);
        }
        String desc = "";
        for(int i=0; i<descriptions.length; i++){
            desc += descriptions[i].trim();
        }
        if(desc.length()==0) {
            ////System.out.println("Descriptions vides ("+flux.name+" : "+flux.URL+").");
            //throw new RSSParsingException("descriptions vides");
        }

        // Récupération des liens

        String[] liens = new String[articles.length];

        for(int i=0; i<articles.length; i++){
            item = Pattern.compile("<link.*?</link>");
            m = item.matcher(articles[i]);
            String candidat = "";
            if(m.find()){
                candidat = m.group();
            }
            if(candidat.equals("")){
                item = Pattern.compile("<link.*?alternate.*?/>");
                m = item.matcher(articles[i]);
                if(m.find()){
                    candidat = m.group();
                }
                // On remplace l'attribut href par une fermeture de balise
                // pour que la suppression des balises ne supprime pas le lien
                candidat = candidat.replaceAll("href=\\\"", "href=>");
                candidat = candidat.replaceAll("\\\"", "<");
                candidat = candidat.replaceAll("href=\\'", "href=>");
                candidat = candidat.replaceAll("\\'", "<");
            }
            if(candidat.equals("")){
                item = Pattern.compile("<alink.*?/alink>");
                m = item.matcher(articles[i]);
                if(m.find()){
                    candidat = m.group();
                }
            }
            candidat = candidat.replaceAll("<!\\[CDATA\\[", "");
            candidat = candidat.replaceAll("\\]\\]>", "");
            liens[i] = Codecs.deHTMLize(candidat);
        }
        String lien = "";
        for(int i=0; i<liens.length; i++){
            lien += liens[i];
        }
        if(lien.length()==0) {
            //Out.printErreur("Liens vides ("+flux.name+" : "+flux.URL+").");
            //throw new RSSParsingException("pas de lien");
        }

        // Récupération des dates de publication

        String[] dates = new String[articles.length];
        for(int i=0; i<articles.length; i++){
            item = Pattern.compile("<published.*?/published>");
            m = item.matcher(articles[i]);
            String candidat = "";
            if(m.find()){
                candidat = m.group();
            }
            if(candidat.equals("")){
                item = Pattern.compile("<issued.*?/issued>");
                m = item.matcher(articles[i]);
                if(m.find()){
                    candidat = m.group();
                }
            }
            if(candidat.equals("")){
                item = Pattern.compile("<pubDate.*?/pubDate>");
                m = item.matcher(articles[i]);
                if(m.find()){
                    candidat = m.group();
                }
            }
            if(candidat.equals("")){
                item = Pattern.compile("<dc:date.*?/dc:date>");
                m = item.matcher(articles[i]);
                if(m.find()){
                    candidat = m.group();
                }
            }
            if(candidat.equals("")){
                item = Pattern.compile("<updated.*?/updated>");
                m = item.matcher(articles[i]);
                if(m.find()){
                    candidat = m.group();
                }
            }
            candidat = candidat.replaceAll("<!\\[CDATA\\[", "");
            candidat = candidat.replaceAll("\\]\\]>", "");
            dates[i] = Codecs.deHTMLize(candidat).trim();
        }
        String date = "";
        for(int i=0; i<dates.length; i++){
            date += dates[i];
        }
        if(date.length()==0) {
            throw new RSSParsingException("dates non renseignées");
        }

        Date[] dates2 = new Date[dates.length];
        boolean remarqueDate = false;

        for(int i=0; i<dates.length; i++){

            GregorianCalendar d = new GregorianCalendar();
            d.setTimeInMillis(0);
            boolean formatFound = false;
            if(!formatFound){
            // Wed, 21 Apr 2010 08:39:27 EST (Eastern Standard Time, GMT-5)
                item = Pattern.compile("[MTWFS]..,\\s\\d{1,2}\\s[jfmasondJFMASOND].*?\\s\\d{4}\\s\\d{2}:\\d{2}:\\d{2}\\sEST");
                m = item.matcher(dates[i]);
                if (m.find()) {
                    formatFound = true;
                    try{
                        TimeZone t = TimeZone.getTimeZone("UTC");
                        t.setRawOffset(-5*3600000);
                        d.setTimeZone(t);
                        String[] s = dates[i].split("\\s");
                        d.set(GregorianCalendar.DAY_OF_MONTH, Integer.parseInt(s[1]));
                        d.set(GregorianCalendar.MONTH, mois.get(s[2]));
                        d.set(GregorianCalendar.YEAR, Integer.parseInt(s[3]));
                        String[] h = s[4].split(":");
                        d.set(GregorianCalendar.HOUR_OF_DAY, Integer.parseInt(h[0]));
                        d.set(GregorianCalendar.MINUTE, Integer.parseInt(h[1]));
                        d.set(GregorianCalendar.SECOND, Integer.parseInt(h[2]));
                        d.set(GregorianCalendar.MILLISECOND, 0);
                    } catch(Exception e){
                        //System.out.println("[01] "+e.getClass()+" ("+e.getStackTrace()[0].getLineNumber()+") : "+e.getMessage());
                        d = null;
                    }
                }
            }
            if(!formatFound){
            // Wed, 21 Apr 2010 08:39:27 EDT (Eastern Daylight Time, GMT-4)
                item = Pattern.compile("[MTWFS]..,\\s\\d{1,2}\\s[jfmasondJFMASOND].*?\\s\\d{4}\\s\\d{2}:\\d{2}:\\d{2}\\sEDT");
                m = item.matcher(dates[i]);
                if (m.find()) {
                    formatFound = true;
                    try{
                        TimeZone t = TimeZone.getTimeZone("UTC");
                        t.setRawOffset(-4*3600000);
                        d.setTimeZone(t);
                        String[] s = dates[i].split("\\s");
                        d.set(GregorianCalendar.DAY_OF_MONTH, Integer.parseInt(s[1]));
                        d.set(GregorianCalendar.MONTH, mois.get(s[2]));
                        d.set(GregorianCalendar.YEAR, Integer.parseInt(s[3]));
                        String[] h = s[4].split(":");
                        d.set(GregorianCalendar.HOUR_OF_DAY, Integer.parseInt(h[0]));
                        d.set(GregorianCalendar.MINUTE, Integer.parseInt(h[1]));
                        d.set(GregorianCalendar.SECOND, Integer.parseInt(h[2]));
                        d.set(GregorianCalendar.MILLISECOND, 0);
                    } catch(Exception e){
                        //System.out.println("[01] "+e.getClass()+" ("+e.getStackTrace()[0].getLineNumber()+") : "+e.getMessage());
                        d = null;
                    }
                }
            }
            if(!formatFound){
            // Wed, 21 Apr 2010 08:39:27 PDT (Pacific Daylight Time, GMT-7)
                item = Pattern.compile("[MTWFS]..,\\s\\d{1,2}\\s[jfmasondJFMASOND].*?\\s\\d{4}\\s\\d{2}:\\d{2}:\\d{2}\\sPDT");
                m = item.matcher(dates[i]);
                if (m.find()) {
                    formatFound = true;
                    try{
                        TimeZone t = TimeZone.getTimeZone("UTC");
                        t.setRawOffset(-7*3600000);
                        d.setTimeZone(t);
                        String[] s = dates[i].split("\\s");
                        d.set(GregorianCalendar.DAY_OF_MONTH, Integer.parseInt(s[1]));
                        d.set(GregorianCalendar.MONTH, mois.get(s[2]));
                        d.set(GregorianCalendar.YEAR, Integer.parseInt(s[3]));
                        String[] h = s[4].split(":");
                        d.set(GregorianCalendar.HOUR_OF_DAY, Integer.parseInt(h[0]));
                        d.set(GregorianCalendar.MINUTE, Integer.parseInt(h[1]));
                        d.set(GregorianCalendar.SECOND, Integer.parseInt(h[2]));
                        d.set(GregorianCalendar.MILLISECOND, 0);
                    } catch(Exception e){
                        //System.out.println("[02] "+e.getClass()+" ("+e.getStackTrace()[0].getLineNumber()+") : "+e.getMessage());
                        d = null;
                    }
                }
            }
            if(!formatFound){
            // Wed, 21 Apr 2010 08:39:27 PST (Pacific Standard Time, GMT-8)
                item = Pattern.compile("[MTWFS]..,\\s\\d{1,2}\\s[jfmasondJFMASOND].*?\\s\\d{4}\\s\\d{2}:\\d{2}:\\d{2}\\sPST");
                m = item.matcher(dates[i]);
                if (m.find()) {
                    formatFound = true;
                    try{
                        TimeZone t = TimeZone.getTimeZone("UTC");
                        t.setRawOffset(-8*3600000);
                        d.setTimeZone(t);
                        String[] s = dates[i].split("\\s");
                        d.set(GregorianCalendar.DAY_OF_MONTH, Integer.parseInt(s[1]));
                        d.set(GregorianCalendar.MONTH, mois.get(s[2]));
                        d.set(GregorianCalendar.YEAR, Integer.parseInt(s[3]));
                        String[] h = s[4].split(":");
                        d.set(GregorianCalendar.HOUR_OF_DAY, Integer.parseInt(h[0]));
                        d.set(GregorianCalendar.MINUTE, Integer.parseInt(h[1]));
                        d.set(GregorianCalendar.SECOND, Integer.parseInt(h[2]));
                        d.set(GregorianCalendar.MILLISECOND, 0);
                    } catch(Exception e){
                        //System.out.println("[03] "+e.getClass()+" ("+e.getStackTrace()[0].getLineNumber()+") : "+e.getMessage());
                        d = null;
                    }
                }
            }
            if(!formatFound){
            // Wed, 21 Apr 2010 08:39 GMT (Greenwitch Mean Time)
                item = Pattern.compile("[MTWFS]..,\\s\\d{1,2}\\s[jfmasondJFMASOND].*?\\s\\d{4}\\s\\d{2}:\\d{2}\\sGMT");
                m = item.matcher(dates[i]);
                if (m.find()) {
                    formatFound = true;
                    try{
                        TimeZone t = TimeZone.getTimeZone("GMT");
                        d.setTimeZone(t);
                        String[] s = dates[i].split("\\s");
                        d.set(GregorianCalendar.DAY_OF_MONTH, Integer.parseInt(s[1]));
                        d.set(GregorianCalendar.MONTH, mois.get(s[2]));
                        d.set(GregorianCalendar.YEAR, Integer.parseInt(s[3]));
                        String[] h = s[4].split(":");
                        d.set(GregorianCalendar.HOUR_OF_DAY, Integer.parseInt(h[0]));
                        d.set(GregorianCalendar.MINUTE, Integer.parseInt(h[1]));
                        d.set(GregorianCalendar.SECOND, 0);
                        d.set(GregorianCalendar.MILLISECOND, 0);
                    } catch(Exception e){
                        //System.out.println("[04] "+e.getClass()+" ("+e.getStackTrace()[0].getLineNumber()+") : "+e.getMessage());
                        d = null;
                    }
                }
            }
            if(!formatFound){
            // Wed, 21 Apr 2010 08:39:27 GMT
                item = Pattern.compile("[MTWFS]..,\\s\\d{1,2}\\s[jfmasondJFMASOND].*?\\s\\d{4}\\s\\d{2}:\\d{2}:\\d{2}\\sGMT");
                m = item.matcher(dates[i]);
                if (m.find()) {
                    formatFound = true;
                    try{
                        TimeZone t = TimeZone.getTimeZone("GMT");
                        d.setTimeZone(t);
                        String[] s = dates[i].split("\\s");
                        d.set(GregorianCalendar.DAY_OF_MONTH, Integer.parseInt(s[1]));
                        d.set(GregorianCalendar.MONTH, mois.get(s[2]));
                        d.set(GregorianCalendar.YEAR, Integer.parseInt(s[3]));
                        String[] h = s[4].split(":");
                        d.set(GregorianCalendar.HOUR_OF_DAY, Integer.parseInt(h[0]));
                        d.set(GregorianCalendar.MINUTE, Integer.parseInt(h[1]));
                        d.set(GregorianCalendar.SECOND, Integer.parseInt(h[2]));
                        d.set(GregorianCalendar.MILLISECOND, 0);
                    } catch(Exception e){
                        //System.out.println("[05] "+e.getClass()+" ("+e.getStackTrace()[0].getLineNumber()+") : "+e.getMessage());
                        d = null;
                    }
                }
            }
            if(!formatFound){
            // Wed, 21 Apr 2010 08:39:27 CEST (Central European Standard Time)
                item = Pattern.compile("[MTWFS]..,\\s\\d{1,2}\\s[jfmasondJFMASOND].*?\\s\\d{4}\\s\\d{2}:\\d{2}:\\d{2}\\sCEST");
                m = item.matcher(dates[i]);
                if (m.find()) {
                    formatFound = true;
                    try{
                        TimeZone t = TimeZone.getTimeZone("UTC");
                        t.setRawOffset(2*3600000);
                        d.setTimeZone(t);
                        String[] s = dates[i].split("\\s");
                        d.set(GregorianCalendar.DAY_OF_MONTH, Integer.parseInt(s[1]));
                        d.set(GregorianCalendar.MONTH, mois.get(s[2]));
                        d.set(GregorianCalendar.YEAR, Integer.parseInt(s[3]));
                        String[] h = s[4].split(":");
                        d.set(GregorianCalendar.HOUR_OF_DAY, Integer.parseInt(h[0]));
                        d.set(GregorianCalendar.MINUTE, Integer.parseInt(h[1]));
                        d.set(GregorianCalendar.SECOND, Integer.parseInt(h[2]));
                        d.set(GregorianCalendar.MILLISECOND, 0);
                    } catch(Exception e){
                        //System.out.println("[06] "+e.getClass()+" ("+e.getStackTrace()[0].getLineNumber()+") : "+e.getMessage());
                        d = null;
                    }
                }
            }
            if(!formatFound){
            // Wed, 21 Apr 2010 08:39:27 CET (Central European Time)
                item = Pattern.compile("[MTWFS]..,\\s\\d{1,2}\\s[jfmasondJFMASOND].*?\\s\\d{4}\\s\\d{2}:\\d{2}:\\d{2}\\sCET");
                m = item.matcher(dates[i]);
                if (m.find()) {
                    formatFound = true;
                    try{
                        TimeZone t = TimeZone.getTimeZone("UTC");
                        t.setRawOffset(1*3600000);
                        d.setTimeZone(t);
                        String[] s = dates[i].split("\\s");
                        d.set(GregorianCalendar.DAY_OF_MONTH, Integer.parseInt(s[1]));
                        d.set(GregorianCalendar.MONTH, mois.get(s[2]));
                        d.set(GregorianCalendar.YEAR, Integer.parseInt(s[3]));
                        String[] h = s[4].split(":");
                        d.set(GregorianCalendar.HOUR_OF_DAY, Integer.parseInt(h[0]));
                        d.set(GregorianCalendar.MINUTE, Integer.parseInt(h[1]));
                        d.set(GregorianCalendar.SECOND, Integer.parseInt(h[2]));
                        d.set(GregorianCalendar.MILLISECOND, 0);
                    } catch(Exception e){
                        //System.out.println("[07] "+e.getClass()+" ("+e.getStackTrace()[0].getLineNumber()+") : "+e.getMessage());
                        d = null;
                    }
                }
            }
            if(!formatFound){
            // Wed, 21 Apr 2010 08:39:27 am
            // Wed, 21 Apr 2010 08:39:27 pm
                item = Pattern.compile("[MTWFS]..,\\s\\d{1,2}\\s[jfmasondJFMASOND].*?\\s\\d{4}\\s\\d{2}:\\d{2}:\\d{2}\\s[ap]m");
                m = item.matcher(dates[i]);
                if (m.find()) {
                    formatFound = true;
                    try{
                        String[] s = dates[i].split("\\s");
                        d.set(GregorianCalendar.DAY_OF_MONTH, Integer.parseInt(s[1]));
                        d.set(GregorianCalendar.MONTH, mois.get(s[2]));
                        d.set(GregorianCalendar.YEAR, Integer.parseInt(s[3]));
                        String[] h = s[4].split(":");
                        d.set(GregorianCalendar.HOUR, Integer.parseInt(h[0]));
                        d.set(GregorianCalendar.MINUTE, Integer.parseInt(h[1]));
                        d.set(GregorianCalendar.SECOND, Integer.parseInt(h[2]));
                        d.set(GregorianCalendar.MILLISECOND, 0);
                        if(s[5].equalsIgnoreCase("AM")){
                            d.set(GregorianCalendar.AM_PM, GregorianCalendar.AM);
                        } else {
                            d.set(GregorianCalendar.AM_PM, GregorianCalendar.PM);
                        }
                    } catch(Exception e){
                        //System.out.println("[08] "+e.getClass()+" ("+e.getStackTrace()[0].getLineNumber()+") : "+e.getMessage());
                        d = null;
                    }
                }
            }
            if(!formatFound){
            // Wed, 21 Apr 2010 08:39:27 +0x00
            // Wed, 21 Apr 2010 08:39:27 -0x00
            // Wed, 21 Apr 2010 08:39:27  +0x00
            // Wed, 21 Apr 2010 08:39:27  -0x00
                item = Pattern.compile("[MTWFS]..,\\s\\d{1,2}\\s[jfmasondJFMASOND].*?\\s\\d{4}\\s\\d{2}:\\d{2}:\\d{2}\\s{1,}[\\+\\-]0\\d00");
                m = item.matcher(dates[i]);
                if (m.find()) {
                    formatFound = true;
                    try{
                        String[] s = dates[i].split("\\s{1,2}");
                        TimeZone t = TimeZone.getTimeZone("UTC");
                        int x = 0;
                        if(s[5].startsWith("+")){
                            x = Integer.parseInt(s[5].substring(2, 3));
                        } else {
                            x = -Integer.parseInt(s[5].substring(2, 3));
                        }
                        t.setRawOffset(x*3600000);
                        d.setTimeZone(t);
                        d.set(GregorianCalendar.DAY_OF_MONTH, Integer.parseInt(s[1]));
                        d.set(GregorianCalendar.MONTH, mois.get(s[2]));
                        d.set(GregorianCalendar.YEAR, Integer.parseInt(s[3]));
                        String[] h = s[4].split(":");
                        d.set(GregorianCalendar.HOUR_OF_DAY, Integer.parseInt(h[0]));
                        d.set(GregorianCalendar.MINUTE, Integer.parseInt(h[1]));
                        d.set(GregorianCalendar.SECOND, Integer.parseInt(h[2]));
                        d.set(GregorianCalendar.MILLISECOND, 0);
                    } catch(Exception e){
                        //System.out.println("[09] "+e.getClass()+" ("+e.getStackTrace()[0].getLineNumber()+") : "+e.getMessage());
                        d = null;
                    }
                }
            }
            if(!formatFound){
            // Wed, 21 Apr 2010 08:39:27 +0x:00
            // Wed, 21 Apr 2010 08:39:27 -0x:00
                item = Pattern.compile("[MTWFS]..,\\s\\d{1,2}\\s[jfmasondJFMASOND].*?\\s\\d{4}\\s\\d{2}:\\d{2}:\\d{2}\\s[\\+\\-]0\\d:00");
                m = item.matcher(dates[i]);
                if (m.find()) {
                    formatFound = true;
                    try{
                        String[] s = dates[i].split("\\s{1,2}");
                        TimeZone t = TimeZone.getTimeZone("UTC");
                        int x = 0;
                        if(s[5].startsWith("+")){
                            x = Integer.parseInt(s[5].substring(2, 3));
                        } else {
                            x = -Integer.parseInt(s[5].substring(2, 3));
                        }
                        t.setRawOffset(x*3600000);
                        d.setTimeZone(t);
                        d.set(GregorianCalendar.DAY_OF_MONTH, Integer.parseInt(s[1]));
                        d.set(GregorianCalendar.MONTH, mois.get(s[2]));
                        d.set(GregorianCalendar.YEAR, Integer.parseInt(s[3]));
                        String[] h = s[4].split(":");
                        d.set(GregorianCalendar.HOUR_OF_DAY, Integer.parseInt(h[0]));
                        d.set(GregorianCalendar.MINUTE, Integer.parseInt(h[1]));
                        d.set(GregorianCalendar.SECOND, Integer.parseInt(h[2]));
                        d.set(GregorianCalendar.MILLISECOND, 0);
                    } catch(Exception e){
                        //System.out.println("[10] "+e.getClass()+" ("+e.getStackTrace()[0].getLineNumber()+") : "+e.getMessage());
                        d = null;
                    }
                }
            }
            if(!formatFound){
            // 21 Apr 2010 08:39:27 +0x00
                item = Pattern.compile("\\d{1,2}\\s[jfmasondJFMASOND].*?\\s\\d{4}\\s\\d{2}:\\d{2}:\\d{2}\\s{1,}[\\+\\-]0\\d00");
                m = item.matcher(dates[i]);
                if (m.find()) {
                    formatFound = true;
                    try{
                        String[] s = dates[i].split("\\s{1,2}");
                        TimeZone t = TimeZone.getTimeZone("UTC");
                        int x = 0;
                        if(s[4].startsWith("+")){
                            x = Integer.parseInt(s[4].substring(2, 3));
                        } else {
                            x = -Integer.parseInt(s[4].substring(2, 3));
                        }
                        t.setRawOffset(x*3600000);
                        d.setTimeZone(t);
                        d.set(GregorianCalendar.DAY_OF_MONTH, Integer.parseInt(s[0]));
                        d.set(GregorianCalendar.MONTH, mois.get(s[1]));
                        d.set(GregorianCalendar.YEAR, Integer.parseInt(s[2]));
                        String[] h = s[3].split(":");
                        d.set(GregorianCalendar.HOUR_OF_DAY, Integer.parseInt(h[0]));
                        d.set(GregorianCalendar.MINUTE, Integer.parseInt(h[1]));
                        d.set(GregorianCalendar.SECOND, Integer.parseInt(h[2]));
                        d.set(GregorianCalendar.MILLISECOND, 0);
                    } catch(Exception e){
                        //System.out.println("[09] "+e.getClass()+" ("+e.getStackTrace()[0].getLineNumber()+") : "+e.getMessage());
                        d = null;
                    }
                }
            }
            if(!formatFound){
            // mardi 20 avril 2010, 18h58
                item = Pattern.compile("[lmjvsd].*?\\s\\d{1,2}\\s[jfmasondJFMASOND].*?\\s\\d{4},\\s\\d{1,2}h\\d{2}");
                m = item.matcher(dates[i]);
                if (m.find()) {
                    formatFound = true;
                    try{
                        String[] s = dates[i].split("\\s{1,2}");
                        d.set(GregorianCalendar.DAY_OF_MONTH, Integer.parseInt(s[1]));
                        d.set(GregorianCalendar.MONTH, mois.get(s[2]));
                        d.set(GregorianCalendar.YEAR, Integer.parseInt(s[3].replaceAll(",", "")));
                        String[] h = s[4].split("h");
                        d.set(GregorianCalendar.HOUR_OF_DAY, Integer.parseInt(h[0]));
                        d.set(GregorianCalendar.MINUTE, Integer.parseInt(h[1]));
                        d.set(GregorianCalendar.SECOND, 0);
                        d.set(GregorianCalendar.MILLISECOND, 0);
                    } catch(Exception e){
                        //System.out.println("[11] "+e.getClass()+" ("+e.getStackTrace()[0].getLineNumber()+") : "+e.getMessage());
                        d = null;
                    }
                }
            }
            if(!formatFound){
            // 2010-04-21T09:09:14+0x:00
            // 2010-04-21T09:09:14-0x:00
                item = Pattern.compile("\\d{4}-\\d{2}-\\d{2}T\\d{2}:\\d{2}:\\d{2}[\\+\\-]0\\d:00");
                m = item.matcher(dates[i]);
                if (m.find()) {
                    formatFound = true;
                    try{
                        TimeZone t = TimeZone.getTimeZone("UTC");
                        int x = 0;
                        if(dates[i].charAt(19)=='+'){
                            x = Integer.parseInt(dates[i].substring(21, 22));
                        } else {
                            x = -Integer.parseInt(dates[i].substring(21, 22));
                        }
                        t.setRawOffset(x*3600000);
                        d.setTimeZone(t);
                        String[] s = dates[i].split("[\\-:T+]");
                        d.set(GregorianCalendar.DAY_OF_MONTH, Integer.parseInt(s[2]));
                        d.set(GregorianCalendar.MONTH, Integer.parseInt(s[1])-1);
                        d.set(GregorianCalendar.YEAR, Integer.parseInt(s[0]));
                        d.set(GregorianCalendar.HOUR_OF_DAY, Integer.parseInt(s[3]));
                        d.set(GregorianCalendar.MINUTE, Integer.parseInt(s[4]));
                        d.set(GregorianCalendar.SECOND, Integer.parseInt(s[5]));
                        d.set(GregorianCalendar.MILLISECOND, 0);
                    } catch(Exception e){
                        //System.out.println("[12] "+e.getClass()+" ("+e.getStackTrace()[0].getLineNumber()+") : "+e.getMessage());
                        d = null;
                    }
                }
            }
            if(!formatFound){
            // 2010-04-21T09:09+0x:00
            // 2010-04-21T09:09-0x:00
                item = Pattern.compile("\\d{4}-\\d{2}-\\d{2}T\\d{2}:\\d{2}[\\+\\-]0\\d:00");
                m = item.matcher(dates[i]);
                if (m.find()) {
                    formatFound = true;
                    try{
                        TimeZone t = TimeZone.getTimeZone("UTC");
                        int x = 0;
                        if(dates[i].charAt(17)=='+'){
                            x = Integer.parseInt(dates[i].substring(18, 19));
                        } else {
                            x = -Integer.parseInt(dates[i].substring(18, 19));
                        }
                        t.setRawOffset(x*3600000);
                        d.setTimeZone(t);
                        String[] s = dates[i].split("[\\-:T+]");
                        d.set(GregorianCalendar.DAY_OF_MONTH, Integer.parseInt(s[2]));
                        d.set(GregorianCalendar.MONTH, Integer.parseInt(s[1])-1);
                        d.set(GregorianCalendar.YEAR, Integer.parseInt(s[0]));
                        d.set(GregorianCalendar.HOUR_OF_DAY, Integer.parseInt(s[3]));
                        d.set(GregorianCalendar.MINUTE, Integer.parseInt(s[4]));
                        d.set(GregorianCalendar.SECOND, 0);
                        d.set(GregorianCalendar.MILLISECOND, 0);
                    } catch(Exception e){
                        //System.out.println("[13] "+e.getClass()+" ("+e.getStackTrace()[0].getLineNumber()+") : "+e.getMessage());
                        d = null;
                    }
                }
            }
            if(!formatFound){
            // 2010-04-21T09:09:14Z (Z = GMT)
                item = Pattern.compile("\\d{4}-\\d{2}-\\d{2}T\\d{2}:\\d{2}:\\d{2}Z");
                m = item.matcher(dates[i]);
                if (m.find()) {
                    formatFound = true;
                    try{
                        TimeZone t = TimeZone.getTimeZone("GMT");
                        d.setTimeZone(t);
                        String[] s = dates[i].split("[\\-:TZ]");
                        d.set(GregorianCalendar.DAY_OF_MONTH, Integer.parseInt(s[2]));
                        d.set(GregorianCalendar.MONTH, Integer.parseInt(s[1])-1);
                        d.set(GregorianCalendar.YEAR, Integer.parseInt(s[0]));
                        d.set(GregorianCalendar.HOUR_OF_DAY, Integer.parseInt(s[3]));
                        d.set(GregorianCalendar.MINUTE, Integer.parseInt(s[4]));
                        d.set(GregorianCalendar.SECOND, Integer.parseInt(s[5]));
                        d.set(GregorianCalendar.MILLISECOND, 0);
                    } catch(Exception e){
                        //System.out.println("[14] "+e.getClass()+" ("+e.getStackTrace()[0].getLineNumber()+") : "+e.getMessage());
                        d = null;
                    }
                }
            }
            if(!formatFound){
            // 2010-04-21T09:09:14.000+02:00
            // 2010-04-21T09:09:14.000-02:00
                item = Pattern.compile("\\d{4}-\\d{2}-\\d{2}T\\d{2}:\\d{2}:\\d{2}\\.\\d{3}[\\+\\-]0\\d:00");
                m = item.matcher(dates[i]);
                if (m.find()) {
                    formatFound = true;
                    try{
                        TimeZone t = TimeZone.getTimeZone("UTC");
                        int x = 0;
                        if(dates[i].charAt(24)=='+'){
                            x = Integer.parseInt(dates[i].substring(25, 26));
                        } else {
                            x = -Integer.parseInt(dates[i].substring(25, 26));
                        }
                        t.setRawOffset(x*3600000);
                        d.setTimeZone(t);
                        String[] s = dates[i].split("[\\-:T\\.]");
                        d.set(GregorianCalendar.DAY_OF_MONTH, Integer.parseInt(s[2]));
                        d.set(GregorianCalendar.MONTH, Integer.parseInt(s[1])-1);
                        d.set(GregorianCalendar.YEAR, Integer.parseInt(s[0]));
                        d.set(GregorianCalendar.HOUR_OF_DAY, Integer.parseInt(s[3]));
                        d.set(GregorianCalendar.MINUTE, Integer.parseInt(s[4]));
                        d.set(GregorianCalendar.SECOND, Integer.parseInt(s[5]));
                        d.set(GregorianCalendar.MILLISECOND, 0);
                    } catch(Exception e){
                        //System.out.println("[15] "+e.getClass()+" ("+e.getStackTrace()[0].getLineNumber()+") : "+e.getMessage());
                        d = null;
                    }
                }
            }
            if(!formatFound){
            // 2010-04-21T09:09:14
                item = Pattern.compile("\\d{4}-\\d{2}-\\d{2}T\\d{2}:\\d{2}:\\d{2}");
                m = item.matcher(dates[i]);
                if (m.find()) {
                    formatFound = true;
                    try{
                        String[] s = dates[i].split("[\\-:T]");
                        d.set(GregorianCalendar.DAY_OF_MONTH, Integer.parseInt(s[2]));
                        d.set(GregorianCalendar.MONTH, Integer.parseInt(s[1])-1);
                        d.set(GregorianCalendar.YEAR, Integer.parseInt(s[0]));
                        d.set(GregorianCalendar.HOUR_OF_DAY, Integer.parseInt(s[3]));
                        d.set(GregorianCalendar.MINUTE, Integer.parseInt(s[4]));
                        d.set(GregorianCalendar.SECOND, Integer.parseInt(s[5]));
                        d.set(GregorianCalendar.MILLISECOND, 0);
                    } catch(Exception e){
                        //System.out.println("[15] "+e.getClass()+" ("+e.getStackTrace()[0].getLineNumber()+") : "+e.getMessage());
                        d = null;
                    }
                }
            }
            if(!formatFound){
            // 1270992580 (nombre de millisecondes depuis le 1/1/1970)
                item = Pattern.compile("\\d{10}");
                m = item.matcher(dates[i]);
                if (m.find()) {
                    formatFound = true;
                    try{
                        d.setTimeInMillis(Integer.parseInt(dates[i]));
                    } catch(Exception e){
                        //System.out.println("[16] "+e.getClass()+" ("+e.getStackTrace()[0].getLineNumber()+") : "+e.getMessage());
                        d = null;
                    }
                }
            }
            if(!formatFound){
            // 2010-04-21 : incorrect (pas l'heure)
                item = Pattern.compile("\\d{4}-\\d{2}-\\d{2}");
                m = item.matcher(dates[i]);
                if (m.find()) {
                    formatFound = true;
                    d = null;
                }
            }
            if(!formatFound){
            // Wed, 21 Apr 2010 +0x:00 : incorrect (pas l'heure)
                item = Pattern.compile("[MTWFS]..,\\s\\d{1,2}\\s[jfmasondJFMASOND].*?\\s\\d{4}\\s[\\+\\-]0\\d:00");
                m = item.matcher(dates[i]);
                if (m.find()) {
                    formatFound = true;
                    d = null;
                }
            }
            if(!formatFound){
            // date vide
                if(dates[i].equals("")){
                    formatFound = true;
                    d = null;
                }
            }
            if(!formatFound){
                d = null;
                throw new RSSParsingException("format de date inconnu : "+dates[i]);
            }
            if(d != null){
                dates2[i] = new Date(d.getTimeInMillis());
            } else {
                // Date insuffisante
                dates2[i] = null;
                if(!remarqueDate){
                    Out.printErreur("Flux "+flux.name+" - date insuffisante : '"+dates[i]+"'");
                }
                remarqueDate = true;
            }
        }

        Vector<BaseRSSItem> res0 = new Vector<BaseRSSItem>();
        for(int i=0; i<articles.length; i++){
            String titre0 = titres[i].trim();
            String desc0 = descriptions[i].trim();
            String lien0 = liens[i].trim();
            Date date0 = dates2[i];
            if(date0 != null && !titre0.equals("")){
                BaseRSSItem item0 = new BaseRSSItem();
                item0.setTitle(titre0);
                item0.setDescription(desc0);
                item0.setLink(lien0);
                item0.setPubDate(date0);
                item0.setIdFluxRSS(Integer.parseInt(flux.ID));
                res0.addElement(item0);
            }
        }
        BaseRSSItem[] res = new BaseRSSItem[res0.size()];
        res0.toArray(res);

        Arrays.sort(res, new ComparateurBaseRSSItem());
        return res;
    }

    /*private static void afficheDate(String date, GregorianCalendar d){
        GregorianCalendar d2 = new GregorianCalendar();
        d2.setTimeInMillis(d.getTimeInMillis());
        if (d2.getTimeInMillis() > 0) {
            //System.out.println(date + " ==> " +
                    d2.get(d2.DAY_OF_MONTH) + "/" + (d2.get(d2.MONTH)+1) + "/" + d2.get(d2.YEAR) + " " +
                    d2.get(d2.HOUR_OF_DAY) + ":" + d2.get(d2.MINUTE) + ":" + d2.get(d2.SECOND));
        }
    }*/
    

    public static BaseRSSItem[] lectureRome(RSSFeedInfoExtended flux) throws RSSParsingException {
        BaseRSSItem[] res;
        RSSFeed rf = new RSSFeed();
        if (rf.InitRSS(flux.URL) == RSSFeed.RSS_OK) {
            //for the given feed, insert the new items into the database;
            SyndEntryImpl[] entryList = rf.GetEntries();
            if (entryList != null) {
                res = new BaseRSSItem[entryList.length];
                for(int i=0; i<entryList.length; i++){
                    res[i] = new BaseRSSItem();
                    res[i].setIdFluxRSS(Integer.parseInt(flux.ID));
                    res[i].setTitle(Codecs.deHTMLize(entryList[i].getTitle()));
                    String desc = "";
                    if (entryList[i].getDescription() != null) {
                        desc = entryList[i].getDescription().getValue();
                    } else if (entryList[i].getContents().size() > 0) {
                        desc = ((SyndContentImpl) (entryList[i].getContents().get(0))).getValue();
                    } else {
                        //On n'y peut rien, là.
                    }
                    res[i].setDescription(Codecs.deHTMLize(desc));
                    String url = entryList[i].getLink();
                    if (url == null) {
                        url = "";
                    }
                    res[i].setLink(url);
                    /*if(url.equals("")){
                        Out.printErreur("Pas d'URL : "+res[i].getTitle()+" ("+flux.name+")");
                    } else {
                        Out.printInfo("URL : "+url+" ("+res[i].getTitle()+", "+flux.name+")");
                    }*/
                    Date d = entryList[i].getPublishedDate();
                    if(d==null || d.getTime()==0){
                        throw new RSSParsingException("dates non renseignées");
                    }
                    res[i].setPubDate(d);
                }
            } else {
                res = new BaseRSSItem[]{};
            }

        } // if problems with the feed
        else {
            throw new RSSParsingException("flux illisible");
        }
        Arrays.sort(res, new ComparateurBaseRSSItem());
        return res;
    }

    public static BaseRSSItem[] lectureNews(String flux) throws RSSParsingException {
        BaseRSSItem[] res;
        RSSFeed rf = new RSSFeed();
        if (rf.InitRSS(flux) == RSSFeed.RSS_OK) {
            SyndEntryImpl[] entryList = rf.GetEntries();
            if (entryList != null) {
                res = new BaseRSSItem[entryList.length];
                for(int i=0; i<entryList.length; i++){
                    res[i] = new BaseRSSItem();
                    res[i].setIdFluxRSS(0); // l'ID du flux est non pertinent
                    res[i].setTitle(Codecs.deHTMLize(entryList[i].getTitle()));
                    String desc = "";
                    if (entryList[i].getDescription() != null) {
                        desc = entryList[i].getDescription().getValue();
                    } else if (entryList[i].getContents().size() > 0) {
                        desc = ((SyndContentImpl) (entryList[i].getContents().get(0))).getValue();
                    } else {
                        //On n'y peut rien, là.
                    }
                    /*Out.printInfo(Codecs.desEscapeHTML(desc));
                    Out.printInfo("=>");
                    Out.printInfo(desc);
                    Out.printInfo("");
                    Out.printInfo("");*/
                    res[i].setDescription(Codecs.desEscapeHTMLText(desc));
                    String url = entryList[i].getLink();
                    if (url == null) {
                        url = "";
                    }
                    res[i].setLink(url);
                    Date d = entryList[i].getPublishedDate();
                    if(d==null || d.getTime()==0){
                        throw new RSSParsingException("dates non renseignées");
                    }
                    res[i].setPubDate(d);
                }
            } else {
                res = new BaseRSSItem[]{};
            }

        } // if problems with the feed
        else {
            throw new RSSParsingException("flux illisible");
        }
        Arrays.sort(res, new ComparateurBaseRSSItem());
        return res;
    }

    public static BaseRSSItem[] trouveArticlesSujet(String codeSujet) throws RSSParsingException {
        String flux = "http://news.google.fr/news?cf=all&ned=fr&hl=fr&topic=h&" +
                "num=1000&ncl="+codeSujet+"&cf=all&output=rss";
        BaseRSSItem[] res;
        RSSFeed rf = new RSSFeed();
        if (rf.InitRSS(flux) == RSSFeed.RSS_OK) {
            SyndEntryImpl[] entryList = rf.GetEntries();
            if (entryList != null) {
                res = new BaseRSSItem[entryList.length];
                for(int i=0; i<entryList.length; i++){
                    res[i] = new BaseRSSItem();
                    res[i].setIdFluxRSS(0); // l'ID du flux est non pertinent
                    res[i].setTitle(Codecs.deHTMLize(entryList[i].getTitle()));
                    String desc = "";
                    if (entryList[i].getDescription() != null) {
                        desc = entryList[i].getDescription().getValue();
                    } else if (entryList[i].getContents().size() > 0) {
                        desc = ((SyndContentImpl) (entryList[i].getContents().get(0))).getValue();
                    } else {
                        //On n'y peut rien, là.
                    }
                    desc = Codecs.deHTMLize(XMLCleaner.xmlToText(desc));
                    /*Out.printInfo(Codecs.desEscapeHTML(desc));
                    Out.printInfo("=>");
                    Out.printInfo(desc);
                    Out.printInfo("");
                    Out.printInfo("");*/
                    res[i].setDescription(Codecs.desEscapeHTMLText(desc));
                    String url = "";
                    try{
                        url = entryList[i].getUri().split("=")[1];
                    } catch(ArrayIndexOutOfBoundsException aioobe){
                        
                    }
                    if (url == null) {
                        url = "";
                    }
                    res[i].setLink(url);
                }
            } else {
                res = new BaseRSSItem[]{};
            }

        } // if problems with the feed
        else {
            throw new RSSParsingException("flux illisible");
        }
        /*Vector<String> res3 = new Vector<String>();
        for(int i=0; i<res.length; i++){
            if(!(res[i].getLink().equals(""))){
                res3.addElement(res[i].getLink());
            }
        }
        BaseRSSItem[] res2 = new BaseRSSItem[res3.size()];
        res3.toArray(res2);*/
        return res;
    }
}

class ComparateurBaseRSSItem implements Comparator {

    public int compare(Object arg0, Object arg1) {
        BaseRSSItem i0 = (BaseRSSItem) arg0;
        BaseRSSItem i1 = (BaseRSSItem) arg1;
        // tri croissant
        return (int) (i0.getPubDate().getTime() - i1.getPubDate().getTime());
    }
}
